 #include "GaussianTransform.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <complex>
#include <math.h>
#include <iostream>

using namespace std;

const double pi = 3.1415926535898;
const double pi2 = 3.1415926535898*2;

#define COMPLEX(x) (complex<double>(cos(x),sin(x)))

GaussianTransform::GaussianTransform()
{
	// the fourier coefficients of the cosine
	// part of a fourier approximation
	// Calculated in Mathematica
	coefficients = new double[21];

	coefficients[0] = 0.17724538509027776;
	coefficients[1] = 0.16058751919761755;
	coefficients[2] = 0.11943245158706159;
	coefficients[3] = 0.07291327584698129;
	coefficients[4] = 0.03653966652950081;
	coefficients[5] = 0.01503129000348615;
	coefficients[6] = 0.005075766482067776;
	coefficients[7] = 0.0014069566572040906;
	coefficients[8] = 0.00032013525888625747;
	coefficients[9] = 0.00005979433036599211;
	coefficients[10] = 9.167695857412845e-6;
	coefficients[11] = 1.1538096956382893e-6;
	coefficients[12] = 1.192013233098331e-7;
	coefficients[13] = 1.0109047768672672e-8;
	coefficients[14] = 7.03559202493707889866650e-10;
	coefficients[15] = 4.036144431572005804566e-11;
	coefficients[16] = 1.74680314903618438539e-12;
	coefficients[17] = 2.037100094015575545537e-13;
	coefficients[18] = -1.20886567190516610905e-13;
	coefficients[19] = 1.15853165943079646014e-13;
	coefficients[20] = -1.0889540937313515502076708599309284e-13;

	L=10;//interval length

}

GaussianTransform::~GaussianTransform()
{
}

/*

  * given an array of the form :
  * [x_coordinate] [value]
  * prints out x_coordinate value
  *
  * also takes number of target points
*/
void GaussianTransform::output_results()
{

	for(int i = 0; i < Ntargets; i++)
		printf("point,result: %lf %0.16lf\n",targets[i],results[i]);

}

/*

  * Calculates the Gaussian transform directly
  *
  *
*/

void GaussianTransform::slow_gt()
{
	int i,j;
	
	for(i = 0; i < Ntargets; i++)
	{
		results[i] = 0;
		for(j = 0; j < Nsources; j++)
			results[i] += weights[j] * (exp( (-pow(targets[i] - sources[j],2))));
	}
}


/*

  * Calculates the Gaussian transform using the Fourier approximation
  *
  *
*/
void GaussianTransform::trunc_gt(int size)
{
	int i,j;
	for(i = 0; i < Ntargets; i++)
	{
		results[i] = 0;
		for(j = 0; j < Nsources; j++)
			if(abs(targets[i] - sources[j]) <= size/2.0)
				results[i] += weights[j] * (exp( (-pow(targets[i] - sources[j],2))));
	}
}

/*

  * Calculates the Gaussian transform using the Fourier approximation
  *
  *
*/
void GaussianTransform::appr_gt(int Nterms)
{
	int i,j;
	for(i = 0; i < Ntargets; i++)
	{
		results[i] = 0;
		for(j = 0; j < Nsources; j++)
			if(abs(targets[i] - sources[j]) <= L/2.0)
				results[i] += weights[j] * (evaluateFourier(targets[i] - sources[j],Nterms));
	}
}

/*

  * Calculates the Gaussian transform using the Fourier approximation and the fast method
  *
  *
*/
void GaussianTransform::fast_gt(int Nterms)
{
	int i,j,p;
	int P = Nterms;
	double pi2=2*pi;
	complex<double> temp;
	//find the limits of the interval
	int inf=0;
	while (inf<Nsources)
	{
		if (targets[0]-sources[inf]<L/2) break;
		inf++;
	}
	int sup=inf+1;
	while (sup<Nsources)
	{
		if (sources[sup]-targets[0]>L/2) break;
		sup++;
	}
	//allocate W0
	complex<double> *Wi;
	Wi = new complex<double>[P];
	//compute W0 and G0 simultaneously
	for (p=0; p<P; p++)
	{
		//W0
		Wi[p] = complex<double>(0,0);
		for (j=inf; j<sup; j++)
		{
			temp=COMPLEX(-p*pi2*sources[j]/L);
			Wi[p] += weights[j]*temp;
		}
		Wi[p] *= coefficients[p];
		//G0
		if (p==0) results[0] = Wi[0].real();
		else
		{
			temp=COMPLEX(p*pi2*targets[0]/L);
			results[0] += real(Wi[p] * temp + conj(Wi[p]) * conj(temp));
		}
	}
	//allocate Wi- and Wi+
	complex<double> *Wi_inf, *Wi_sup;
	Wi_inf = new complex<double>[P];
	Wi_sup = new complex<double>[P];
	for(i = 1; i < Ntargets; i++)
	{
		//clear Wi-, Wi+
		for (p=0; p<P; p++)
		{
			Wi_inf[p] = complex<double>(0,0);
			Wi_sup[p] = complex<double>(0,0);
		}
		//compute Wi-, advance inf
		while (inf<Nsources)
		{
			if (targets[i]-sources[inf]>=L/2)
			{
				for (p=0; p<P; p++)
					Wi_inf[p] += weights[inf]*COMPLEX(-p*pi2*sources[inf]/L);
				inf++;
			}else break;
		}
		//compute Wi+, advance sup
		while (sup<Nsources)
		{
			if (sources[sup]-targets[i]<=L/2)
			{
				for (p=0; p<P; p++)
					Wi_sup[p] += weights[sup]*COMPLEX(-p*pi2*sources[sup]/L);
				sup++;
			}else break;
		}
		//premultiply Wi- and Wi+ by Qj
		for (p=0; p<P; p++)
		{
			Wi_inf[p] *= coefficients[p];
			Wi_sup[p] *= coefficients[p];
		}
		//evaluate Gi
		Wi[0] += -Wi_inf[0] + Wi_sup[0];
		results[i] = Wi[0].real();
		for (p=1; p<P; p++)
		{
			Wi[p] += -Wi_inf[p] + Wi_sup[p];
			temp=COMPLEX(p*pi2*targets[i]/L);
			results[i]+= real(Wi[p]*temp+conj(Wi[p])*conj(temp));
		}
	}
}

/*

  * Reads in the data from file gData.txt
  * puts source points into data matrix
  * puts targets into a target array
  *
  * returns the source, target arrays, the number of target points,
  * and the number of source points
  *
  * The input data file should be of the form:
  *
  * number_of_sources number_of_targets
  * Source_point Source_weight
  * Source_point Source_weight
  * ...
  * ...
  * ...
  *
  * Target_point
  * Target_point
  * ...
  * ...
*/
bool GaussianTransform::readData(char *fileName)
{
	FILE *fpInput;

	fpInput=fopen(fileName, "r");
	if (!fpInput) return false;

	fscanf(fpInput,"%d%d",&Nsources, &Ntargets);
	fprintf(stdout,"numSources %d numTargets %d\n",Nsources, Ntargets);

	sources = new double [Nsources];
	weights = new double [Nsources];
	targets = new double [Ntargets];
	results = new double [Ntargets];

	for(int nr=0; nr < Nsources; nr++)
	{
		if (fscanf(fpInput, "%lf%lf", &sources[nr], &weights[nr])<0)
		{
			fprintf(stderr, "Error: ReadSources!\n");
			return false;
		}
	}

	for(int i = 0; i<Ntargets; i++)
	{
		if (fscanf(fpInput, "%lf", &targets[i])<0)
		{
			fprintf(stderr, "Error: ReadTargets!\n");
			return false;
		}
	}

	fclose(fpInput);
	printf("Source data:\n\n");
	for(int q = 0; q < nr; q++)
	{
		fprintf(stderr,"%lf %lf\n",sources[q],weights[q]);
	}
	printf("\n\n");
	return true;
}

/*

  * evaluates the cosine Fourier seriers at point x
  * and includes the first num_terms of the series
  * The fourier coefficients (coefficients []) need to be set before
  * this function is called.
  *
  * coefficients[0] = a0;
  * coefficents [1] = a1, a-1;
  * ....
*/

double GaussianTransform::evaluateFourier(double x,int Nterms)
{
	x = x/L;
	double sum = 0;
	double temp;
	sum = coefficients[0];
	for(int p = 1; p < Nterms; p ++)
	{
		temp=p * pi2 * x;
		sum += coefficients[p] * (cos(temp) + cos(-temp));
	}
	return sum;
}



/*

  * generates numS sources between minS and maxS and perturbs them by sigmaS
  * generates numS weights with randomness sigmaW
  * generates numT targets between minT and maxT and perturbs them by sigmaT
*/

void GaussianTransform::generateData(int minS, int maxS, int numS, int sigmaS, int sigmaW, int minT, int maxT, int numT, int sigmaT)
{
	Nsources=numS;
	Ntargets=numT;
	sources = new double [Nsources];
	weights = new double [Nsources];
	targets = new double [Ntargets];
	results = new double [Ntargets];

	int c=0;
	for (double s=minS; s<=maxS; s+=double(maxS-minS)/numS)
		sources[c++]=s+(double(rand())-RAND_MAX/2)/RAND_MAX*sigmaS;
	c=0;
	for (double t=minT; t<=maxT; t+=double(maxT-minT)/numT)
		targets[c++]=t+(double(rand())-RAND_MAX/2)/RAND_MAX*sigmaT;
	for (c=0; c<numS; c++)
		weights[c]=1+double(rand())/RAND_MAX*sigmaW;
	printf("Source data:\n\n");
	for(int q = 0; q < Nsources; q++)
	{
		fprintf(stderr,"%lf %lf\n",sources[q],weights[q]);
	}
	printf("\n\n");
}


void GaussianTransform::startTimer()
{
	_ftime( &start );
}


void GaussianTransform::stopTimer()
{
	_ftime( &stop );
}


void GaussianTransform::printTime()
{
	double t=(stop.time-start.time)*1000.0+stop.millitm-start.millitm;
	printf("Time:%0.0lf\n",t);
}