﻿/*This class contains methods for translating and working with
Braille characters.*/
//Created May 22, 2007
//Updated July 18, 2007

class Braille implements Observer, InputMode {

	//Arrays for translation
	public static var dots:Array = [null, "f", "d", "s", "j", "k", "l", " "];
	public static var dotsToLetters:Array;;	//Translates dots to characters
	public static var dotsValue:Array;		//Value for each dot
	public static var lettersToNumbers:Array;

	//Private Members
	private var pressedKeys:Array;
	private var keys:Array;					//Selected keys
	private var kl:KeyListener;				//KeyListener object
	private var os:ObservableSubject;		//Allows Braille to be observed
	private var output:String = "";			//Output to the text field
	private var number:Boolean = false;
	private var capital:Boolean = false;
	private var time:Time;
	private var debug:Boolean = false;
	private var openParentheses:Boolean = false;
	private var counter:Number = 0;
	
	//Constructor function
	public function Braille() {
		time = Time.getInstance();
		
		pressedKeys = new Array();
		keys = new Array();
		defineAssociativeArray();
				
		os = new ObservableSubject();

		notifyObservers("Clear output");

		kl = KeyListener.getInstance();
		kl.addObserver(this);
	}
	
	//Update the Braille object by results of listening to various objects	
	public function update(o:Observable, infoObj:Object) {
		
		//Listening to KeyListener object
		if (infoObj == "Key Pressed") {
			keyPressed();
		} 
		else if (infoObj == "Key Released") {
			keyReleased();
		} 
		
		//Listening to SingleKey object
		else if (String(infoObj).substring(0,15) == "Remove this key"){
			var index = findCharInArray(String(infoObj).substring(15,16));
			var releaseTime = keys[index].getReleaseTime();
			if ((index!=-1)&&(releaseTime!=Infinity)){
				keys.splice(index,1);
				setChanged();
				output = translateToLanguage(keys);
			
				if (output == "("){
					if (openParentheses){
						output = ")";
					}
				}
				
				if(debug)
				trace(time.getCurrentTime()+"  "+releaseTime+"  "+String(infoObj).substring(15,16)+
						" has been removed.  Potential Letter = "+output);
				
				notifyObservers("Potential Letter");
			}
		}
	}//end of function
	
	//Action on keypress
	public function keyPressed() {
		var ASCII:Number = kl.getKey();
		//trace(ASCII+"  ("+String.fromCharCode(ASCII)+") has been pressed.");
		
		//The key is not the backspace key
		if ((ASCII != 8)&&(ASCII!=46)) {
			var char:String = String.fromCharCode(ASCII);
			//char = char.toLowerCase();
			if ((!Utilities.keysContain(char, keys)) 
							&& (char != "") 
							&& (Utilities.arrayContains(char, dots))) {
				pressedKeys.push(char);
				
				var newKey = new SingleKey(char);
				keys.push(newKey);
				newKey.addObserver(this);
				
				setChanged();
				output = translateToLanguage(keys);
				
			//The Braille combination is the same for open and closed parentheses
			//This section determines which to output
			if (output == "("){
				if (openParentheses){
					output = ")";
				}
			}
				
				if (debug)
				trace(time.getCurrentTime()+"  "+ASCII+"  ("+String.fromCharCode(ASCII)+
						") has been added.  Potential Letter = "+output
						+".  PressedKeys = "+pressedKeys.length);
				
				notifyObservers("Potential Letter");
			}
			else if (Utilities.keysContain(char,keys)){
				var index = findCharInArray(char);
				keys[index].setReleaseTime(Infinity);
				
				if (!Utilities.arrayContains(char, pressedKeys))
				pressedKeys.push(char);
			}
	
		}
		
		//Backspace
		else {
			setChanged();
			keys=[];
			pressedKeys=[];
			notifyObservers("Backspace");
		}
	}
	
	//Action on keyrelease
	public function keyReleased() {
		var releasedKey:String = String.fromCharCode(kl.getKey());
		
		if ((Utilities.arrayContains(releasedKey, dots)==1)
			&&(Utilities.arrayContains(releasedKey, pressedKeys)==1)){
			var index = Utilities.getIndex(releasedKey, pressedKeys);
			pressedKeys.splice(index,1);
			}
			
		
		if (debug)
		trace(time.getCurrentTime()+"  "+kl.getKey()+"  ("+
			String.fromCharCode(kl.getKey())+
			") has been released.  PressedKeys = "+pressedKeys.length);

			
		if (keys.length==0){
			pressedKeys = [];
			setChanged();
			output = "";
			notifyObservers("Selected Letter");
			return;
		}
		
		//Last key to be released
		if (pressedKeys.length == 0) {
			var translatedChar = translateToLanguage(keys);

			//Add new character to output string if valid
			if ((translatedChar.length<=4) && (translatedChar != undefined)) {
				
				if ((translatedChar!= "^") && (translatedChar!= "#")){
					setChanged();
					output = translatedChar;
					
					//The Braille combination is the same for 
					//open and closed parentheses
					//This section determines which to output
					if (output == "("){
						if (openParentheses){
							output = ")";
							openParentheses = false;
						}
						else
							openParentheses = true;
					}
					
					if (debug)
					trace("*** "+time.getCurrentTime()+"  '"+output+"' has been selected\n");					
					
					notifyObservers("Selected Letter");
					
					capital = false;
					number = false;
				}
				else if (translatedChar == "^"){	//Capitalize next value
					capital = true;
					setChanged();
					output = "Mode = CAPITAL LETTER";
					notifyObservers("Message");
				}
				else {							//Change next value to number
					number = true;
					setChanged();
					output = "Mode = NUMBER";
					notifyObservers("Message");
				}
			}
			
			else{
				trace(translatedChar);
				setChanged();
				output = "";
				notifyObservers("Selected Letter");
			}
			
		keys = [];	
		}
		
		else if (pressedKeys != 0)
		{
			var index = findCharInArray(String.fromCharCode(kl.getKey()));
			keys[index].setReleaseTime();
		}
			
	}
	
	//Returns output to listening text field
	public function getNewOutput():String {
		return output;
	}
	
	//Helper function-finds the index of SingleKey objects in the keys array
	private function findCharInArray(char:String):Number{
		for (var i = 0; i<keys.length; i++){
			if (keys[i].getCharName() == char)
				return i;
		}
		return -1;	
	}
	
	//Goes through keys array and makes a new array of just the charNames
	private function translateObjectToString():Array{
		var newArray:Array = new Array();
		
		for (var i = 0; i<keys.length; i++){
			newArray.push(keys[i].getCharName());
		}
		return newArray;
	}

	
	//Uses the array "keys" to obtain the corresponding character.
	private function translateToLanguage():String {
		var output:String;
		var keys = translateObjectToString();
		keys.sort();
		
		//Finds which dots are selected (1,2,...7), where 7 = space
		var dotsSelected:Array = Utilities.findDots(dots, keys);
		//Passes array of numbered dots and obtains its numeric value
		var dotValue:String = translateDots(dotsSelected);

		//Obtains character from numeric value
		if (dotValue == "0") {
			output = "an unrecognized Braille value";
		} else if (dotValue<10) {
			output = dotsToLetters["00"+dotValue];
		} else if (dotValue<100) {
			output = dotsToLetters["0"+dotValue];
		} else {
			output = dotsToLetters[dotValue];
		}
		
		if (capital == true){					//Capitalize
			output = output.toUpperCase();
		}
		if (number == true){					//Convert to number (if valid)
			if (lettersToNumbers[output]!=undefined)
				output = lettersToNumbers[output];
		}
		
		return output;
	}
	
	//Sets up the arrays of values that are used elsewhere
	private function defineAssociativeArray() {
		dotsToLetters = new Array();
		dotsToLetters["000"] = undefined;//A value that is not in the dotArray
		dotsToLetters["100"] = " ";
		dotsToLetters["001"] = "@";
		dotsToLetters["002"] = "|";//Dot 5
		dotsToLetters["003"] = "/";//Dot 4 5
		dotsToLetters["004"] = "^";//CAPS Dot 6
		dotsToLetters["005"] = "~";//Italics
		dotsToLetters["006"] = "\\";//letter sign
		dotsToLetters["007"] = "`";//Dot 4 5 6
		dotsToLetters["010"] = "a";
		dotsToLetters["011"] = "c";
		dotsToLetters["012"] = "e";
		dotsToLetters["013"] = "d";
		dotsToLetters["014"] = "ch";
		dotsToLetters["015"] = "sh";
		dotsToLetters["016"] = "wh";
		dotsToLetters["017"] = "th";
		dotsToLetters["020"] = ",";
		dotsToLetters["021"] = "i";
		dotsToLetters["022"] = ":";
		dotsToLetters["023"] = "j";
		dotsToLetters["024"] = "en";
		dotsToLetters["025"] = "ow";
		dotsToLetters["026"] = ".";
		dotsToLetters["027"] = "w";
		dotsToLetters["030"] = "b";
		dotsToLetters["031"] = "f";
		dotsToLetters["032"] = "h";
		dotsToLetters["033"] = "g";
		dotsToLetters["034"] = "gh";
		dotsToLetters["035"] = "ed";
		dotsToLetters["036"] = "ou";
		dotsToLetters["037"] = "er";
		dotsToLetters["040"] = "'";
		dotsToLetters["041"] = "st";
		dotsToLetters["042"] = "in";
		dotsToLetters["043"] = "ar";
		dotsToLetters["044"] = "-";
		dotsToLetters["045"] = "ing";
		dotsToLetters["046"] = "\"";	//quotation mark
		dotsToLetters["047"] = "#";
		dotsToLetters["050"] = "k";
		dotsToLetters["051"] = "m";
		dotsToLetters["052"] = "o";
		dotsToLetters["053"] = "n";
		dotsToLetters["054"] = "u";
		dotsToLetters["055"] = "x";
		dotsToLetters["056"] = "z";
		dotsToLetters["057"] = "y";
		dotsToLetters["060"] = ";";
		dotsToLetters["061"] = "s";
		dotsToLetters["062"] = "!";
		dotsToLetters["063"] = "t";
		dotsToLetters["064"] = "?";
		dotsToLetters["065"] = "the";
		dotsToLetters["066"] = "(";
		dotsToLetters["067"] = "with";
		dotsToLetters["070"] = "l";
		dotsToLetters["071"] = "p";
		dotsToLetters["072"] = "r";
		dotsToLetters["073"] = "q";
		dotsToLetters["074"] = "v";
		dotsToLetters["075"] = "and";
		dotsToLetters["076"] = "or";
		dotsToLetters["077"] = "for";

		//These values ensure that each dot array has a unique numeric value.
		dotsValue = new Array();
		dotsValue[1] = 10;
		dotsValue[2] = 20;
		dotsValue[3] = 40;
		dotsValue[4] = 1;
		dotsValue[5] = 2;
		dotsValue[6] = 4;
		dotsValue[7] = 100;
		
		lettersToNumbers = new Array();
		lettersToNumbers["a"] = 1;
		lettersToNumbers["b"] = 2;
		lettersToNumbers["c"] = 3;
		lettersToNumbers["d"] = 4;
		lettersToNumbers["e"] = 5;
		lettersToNumbers["f"] = 6;
		lettersToNumbers["g"] = 7;
		lettersToNumbers["h"] = 8;
		lettersToNumbers["i"] = 9;
		lettersToNumbers["j"] = 0;
	}

	//Translates the array of dots into a numeric value by adding each dot's
	//numeric value.
	function translateDots(dots:Array):String {
		var sum:Number = 0;
		for (var i = 0; i<dots.length; i++) {
			sum += dotsValue[dots[i]];
		}
		return sum.toString();
	}
	
//-------------------------------------------------------------------
	//Allows Braille to be listened to
	public function addObserver(o:Observer):Boolean {
		return os.addObserver(o);
	}
	public function removeObserver(o:Observer):Boolean {
		return os.removeObserver(o);
	}
	public function notifyObservers(infoObj:Object):Void {
		os.notifyObservers(infoObj);
	}
	public function clearObservers():Void {
		os.clearObservers();
	}
	public function setChanged():Void {
		os.setChanged();
	}
	public function clearChanged():Void {
		os.clearChanged();
	}
	public function hasChanged():Boolean {
		return os.hasChanged();
	}
	public function countObservers():Number {
		return os.countObservers();
	}
}