from pyOpenAL import *
from math import exp
import os.path, sys, threading, time

#------------------------------------------------------------------------------------#

class EarconPool:
	#-----------------------------------------------
	#intialize the class by initializing the OpenAL library
	#-----------------------------------------------	
	def __init__(self, max_size, pos_func = None):
		self.max_size = max_size
		self.eids = {}
		self.earcons = []
			
		#initialize the OpenAL library
		alutInit(sys.argv)
		
		#make a bunch of reusable sources with a default
		#positioning callback
		for i in range(self.max_size):
			e = Earcon()
			e.SetPositionFunction(pos_func)
			self.earcons.append(e)
		
		#setup a default listener	
		alListenerfv(AL_POSITION, [0,0,0])
		alListenerfv(AL_VELOCITY, [0,0,0])
		alListenerfv(AL_ORIENTATION, [0,0,-1,0,1,0])
	
	#-----------------------------------------------
	#clean up any created earcons and shut down the library
	#-----------------------------------------------	
	def Delete(self):
		for e in self.earcons:
			e.Delete()
				
		#shut down the library
		alutExit()
	
	#-----------------------------------------------
	#make a 'new' earcon by reusing the ones in the pool
	#-----------------------------------------------
	def New(self, file, stamp=0, owner=None, eid=None):
		for e in self.earcons:
			if e.IsOwned() == False:
				e.SetSound(file)
				e.SetStamp(stamp)
				e.SetOwner(owner)
				e.SetID(eid)
				
				#store a reference to the object by id
				if eid != None:
					self.eids[eid] = e
				
				return e
		
		return None

	#-----------------------------------------------
	#cleanup old earcons
	#-----------------------------------------------			
	def Clean(self, stamp, owner):
		for e in self.earcons:
			if e.GetStamp() < stamp and e.GetOwner() == owner:
				eid = e.GetID()
				if self.eids.has_key(eid):
					del self.eids[eid]
				e.Reset()

	#-----------------------------------------------
	#find an existing earcon by its ID
	#-----------------------------------------------
	def FindID(self, eid):
		try:
			return self.eids[eid]
		except:
			return None
			
	#-----------------------------------------------
	#play a list of earcons asynchornously in another thread
	#and manage their disposal on completion
	#-----------------------------------------------			
	def PlayListAsync(self, earcons, spacing, stamp, owner):
		self.async = EarconThread(earcons, spacing, self.Clean, stamp, owner)
		self.async.start()
	
	def Debug(self):
		i = 0
		for e in self.earcons:
			if e.IsOwned():
				i += 1
		return "There are currently",i,"earcons in use."
	
#------------------------------------------------------------------------------------#
		
class Earcon:
	#-----------------------------------------------
	#initialize by creating a single buffer and source
	#-----------------------------------------------	
	def __init__(self):
		self.owner = None
		self.stamp = 0
		self.eid = None

		self.buff = None		
		self.src = alGenSources(1)

	#-----------------------------------------------
	#delete the allocated buffer and source
	#-----------------------------------------------
	def Delete(self):
		alSourceStop(self.src[0])
 		alSourcei(self.src[0], AL_BUFFER, 0)	
 		alDeleteSources(self.src)
 		if self.buff != None:
			alDeleteBuffers(self.buff)

	#-----------------------------------------------
	#return if the earcon has been claimed
	#-----------------------------------------------
	def IsOwned(self):
		return self.owner != None
	
	#-----------------------------------------------
	#stop the sound immediately
	#-----------------------------------------------				
	def Stop(self):
		alSourceStop(self.src[0])
			
	#-----------------------------------------------
	#play the sound
	#-----------------------------------------------			
	def Play(self):
		alSourcePlay(self.src[0])

	#-----------------------------------------------
	#set the sound file
	#-----------------------------------------------	
	def SetSound(self, file):
		(data, format, freq, loop) = alutWAVFileToString(file)
		
		#clean up the buffer if it exists
		if self.buff != None:
			alSourceStop(self.src[0])
			alSourcei(self.src[0], AL_BUFFER, 0)	
			alDeleteBuffers(self.buff)

		self.buff = alGenBuffers(1)
		alBufferData(self.buff[0], format, data, freq)
		alSourcei(self.src[0], AL_BUFFER, self.buff[0])	
			
	#-----------------------------------------------
	#set if the sound is looping
	#-----------------------------------------------
	def SetLooping(self, bool):
		if bool:
			alSourcei(self.src[0], AL_LOOPING, AL_TRUE)	
		else:
			alSourcei(self.src[0], AL_LOOPING, AL_FALSE)
	
	#-----------------------------------------------
	#set a time stamp on the earcon
	#-----------------------------------------------			
	def SetStamp(self, stamp):
		self.stamp = stamp

	#-----------------------------------------------
	#get the time stamp for the earcon
	#-----------------------------------------------			
	def GetStamp(self):
		return self.stamp
		
	#-----------------------------------------------
	#set a owner of the earcon
	#-----------------------------------------------			
	def SetOwner(self, owner):
		self.owner = owner

	#-----------------------------------------------
	#get the owner of the earcon
	#-----------------------------------------------			
	def GetOwner(self):
		return self.owner
		
	#-----------------------------------------------
	#set the earcon ID
	#-----------------------------------------------			
	def SetID(self, eid):
		self.eid = eid

	#-----------------------------------------------
	#get the earcon ID
	#-----------------------------------------------			
	def GetID(self):
		return self.eid
		
	#-----------------------------------------------
	#get the earcon ID
	#-----------------------------------------------
	def Reset(self):
		self.eid = None
		self.owner = None
		self.stamp = None
		
		#clean up the buffer if it exists
		if self.buff != None:
			alSourceStop(self.src[0])
			alSourcef(self.src[0], AL_GAIN, 1.0)
			alSourcei(self.src[0], AL_BUFFER, 0)	
			alDeleteBuffers(self.buff)
			self.buff = None
		
	#-----------------------------------------------
	#set the source position using a callback function
	#-----------------------------------------------
	def SetPositionFunction(self, func):
		self.func = func
	
	#-----------------------------------------------
	#set the source position
	#-----------------------------------------------	
	def SetPosition(self,x,y,z):
		if self.func != None:
			x,y,z = apply(self.func, [x, y, z])
		alSource3f(self.src[0], AL_POSITION, x, y, z)

	#-----------------------------------------------
	#set the gain of the source
	#-----------------------------------------------	
	def SetGain(self, gain):
		alSourcef(self.src[0], AL_GAIN, gain)

#------------------------------------------------------------------------------------#

class EarconThread(threading.Thread):
	#-----------------------------------------------
	#initialize the thread
	#-----------------------------------------------	
	def __init__(self, earcons, spacing, clean, stamp, owner):
		threading.Thread.__init__(self)
		self.earcons = earcons
		self.spacing = spacing
		self.clean = clean
		self.owner = owner
		self.stamp = stamp

	#-----------------------------------------------
	#run through a list of earcons and play them
	#	with a pause in between
	#-----------------------------------------------	
	def run(self):
		for e in self.earcons:
			e.Play()
			time.sleep(self.spacing)
			
		apply(self.clean, (self.stamp, self.owner))

#------------------------------------------------------------------------------------#
				
if __name__ == '__main__':
	import time
	
	pool = EarconPool(30)
	e = pool.New('reservation.wav')
	e.SetLooping(True)
	e.Play()
	e.SetPosition(5,5,0)
	time.sleep(5)
	e.Stop()
	
	e = pool.New('city.wav')
	e.Play()
	time.sleep(5)