#include "pthread++.h"

#include <iostream>
#include <ctime>
using namespace std;

////////////  Mutex  ////////////

Mutex::Mutex(int type)
{
	pthread_mutexattr_t attr;
	
	pthread_mutexattr_init(&attr);
	pthread_mutexattr_settype(&attr, type);
	
	pthread_mutex_init(&mMutex, &attr);
	
	pthread_mutexattr_destroy(&attr);
}

Mutex::~Mutex()
{
	if(trylock())  // returns non-zero if lock is busy
	{
		cerr<<"WARNING: Destroying a locked mutex!"<<endl;
		unlock();
	}
	
	pthread_mutex_destroy(&mMutex);
}

int Mutex::lock()
{
	return pthread_mutex_lock(&mMutex);
}

int Mutex::trylock()
{
	return pthread_mutex_trylock(&mMutex);
}

int Mutex::unlock()
{
	return pthread_mutex_unlock(&mMutex);
}


////////////  MutexRW  ////////////

MutexRW::MutexRW()
{
	pthread_rwlockattr_t attr;
	
	pthread_rwlockattr_init(&attr);
	
	pthread_rwlock_init(&mRwMutex, &attr);
	
	pthread_rwlockattr_destroy(&attr);
}

MutexRW::~MutexRW()
{
	if(tryWriteLock())
	{
		cerr<<"WARNING: Destroying a locked mutex!"<<endl;
		unlock();
	}
	
	pthread_rwlock_destroy(&mRwMutex);
}

int MutexRW::readLock()
{
	return pthread_rwlock_rdlock(&mRwMutex);
}

int MutexRW::tryReadLock()
{
	return pthread_rwlock_tryrdlock(&mRwMutex);
}

int MutexRW::writeLock()
{
	return pthread_rwlock_wrlock(&mRwMutex);
}

int MutexRW::tryWriteLock()
{
	return pthread_rwlock_trywrlock(&mRwMutex);
}

int MutexRW::lock()
{
	return writeLock();
}

int MutexRW::trylock()
{
	return tryWriteLock();
}

int MutexRW::unlock()
{
	return pthread_rwlock_unlock(&mRwMutex);
}


////////////  LockGuard  ////////////

LockGuard::LockGuard(MutexBase& mutex):mMutexRef(mutex)
{
	mMutexRef.lock();
}

LockGuard::~LockGuard()
{
	mMutexRef.unlock();
}


////////////  LockGuardForRead  ////////////

LockGuardForRead::LockGuardForRead(MutexRW& mutex):mMutexRef(mutex)
{
	mMutexRef.readLock();
}

LockGuardForRead::~LockGuardForRead()
{
	mMutexRef.unlock();
}


////////////  Condition  ////////////

Condition::Condition(Mutex& mutex): mMutexRef(mutex)
{
	pthread_condattr_t attr;
	
	pthread_condattr_init (&attr);
	
	pthread_cond_init(&mCondition, &attr);
	
	pthread_condattr_destroy(&attr);
}

Condition::~Condition()
{	
	pthread_cond_destroy(&mCondition);
}

int Condition::wait()
{
	return pthread_cond_wait(&mCondition, &(mMutexRef.mMutex));
}

int Condition::timedWait(int sec, int nsec)
{
	struct timespec timeToWait;
	timeToWait.tv_sec = (long)(time(0) + sec);
	timeToWait.tv_nsec = nsec;
	
	return pthread_cond_timedwait(&mCondition, &(mMutexRef.mMutex), &timeToWait);
}

int Condition::timedWait(int msec)
{
	struct timespec timeToWait;
	
	// can we use gettimeofday() on Windows? time() doesn't afford very good resolution.

	timeToWait.tv_sec = (long)(time(0) + msec/1000);
	timeToWait.tv_nsec = (msec%1000) * 1000000;  // convert remaining ms to ns
	
	return pthread_cond_timedwait(&mCondition, &(mMutexRef.mMutex), &timeToWait);
}

int Condition::signal()
{
	return pthread_cond_signal(&mCondition);
}

int Condition::broadcast()
{
	return pthread_cond_broadcast(&mCondition);
}


////////////  Thread  ////////////

Thread::Thread(int type):
//	mThread(0),
	mType(type),
	mGo(false),
	mSleepCondition(mSleepMutex), // is this safe??
	mRunCondition(mRunMutex)
{
}

Thread::~Thread()
{
	// Should probably kill the thread.
	// What is the best way to do so?
}

int Thread::create()
{
	pthread_attr_t attr;
	
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, mType);
	
	return pthread_create(&mThread, &attr, Thread::_runHelper, this);
}

int Thread::getID()
{
	return (int)(&mThread);
}

int Thread::run()
{
	mRunMutex.lock();
	mGo = true;
	mRunCondition.signal();
	mRunMutex.unlock();
	
	return 0;
}

int Thread::cancel()
{
	return pthread_cancel(mThread);
}

int Thread::detach()
{
	// not thread safe
	if(mType == PTHREAD_CREATE_JOINABLE)
	{
		return pthread_detach(mThread);
	}
	return 0;
}

int Thread::yield()
{
	return sched_yield();
}

int Thread::join(void** exitDataStore)
{
	return pthread_join(mThread, exitDataStore);
}

bool Thread::isSelf()
{
	return (pthread_equal(mThread, pthread_self())==0)?false:true;
}

bool Thread::operator==(const Thread& rhs)
{
	return (pthread_equal(mThread, rhs.mThread)==0)?false:true;
}

int Thread::sleep(int msec)
{
	// assumes we are running and have mRunMutex.  Add state flag??
	return mRunCondition.timedWait(msec);  // wait where there will be no signal => sleep for specified time
}

void Thread::exit(void* exitData)
{
	mRunMutex.unlock();
	pthread_exit(exitData);
}

void* Thread::_runHelper(void* threadInstance)
{
	((Thread*)threadInstance)->_startingGate();
	
	return NULL;
}

void Thread::_startingGate()
{
	mRunMutex.lock();
	
	if(!mGo)
	{
		mRunCondition.wait();  // wait to be signaled by run()
	}
	
	this->entry();
	
	mRunMutex.unlock();
	
	pthread_exit(0);
}