#ifndef PTHREAD_CPP_H
#define PTHREAD_CPP_H

#include <pthread.h>

//////////////////////

class MutexBase  // interface mutex classes
{
public:
	virtual int lock() = 0;
	virtual int trylock() = 0;
	virtual int unlock() = 0;
};

//////////////////////

class Mutex: public MutexBase
{
	friend class Condition;
public:
	Mutex(int type = PTHREAD_MUTEX_NORMAL);
	~Mutex();

	int lock();
	int trylock();
	int unlock();
	
private:
	pthread_mutex_t mMutex;
};

//////////////////////

// Implementes lock & trylock that wrap writeLock calls.
// This allows us to use MutexRW just like a regular
// mutex if need be (though there might be some extra
// overhead).
class MutexRW: public MutexBase
{
public:
	MutexRW();
	~MutexRW();
	
	int readLock();
	int tryReadLock();
	
	int writeLock();
	int tryWriteLock();
	
	int lock();     // wrapper for writeLock()
	int trylock();  // wrapper for tryWriteLock()
	int unlock();
private:
	pthread_rwlock_t mRwMutex;
};

//////////////////////

// A safe way to use mutexes in functions with multiple exit-points
// or that can throw exceptions.  Instantiate a lock guard and pass
// it the mutex you wish to lock.  It will be unlocked when the guard
// loses scope.
class LockGuard // aka CriticalSection
{
public:
	LockGuard(MutexBase&);
	~LockGuard();
private:
	MutexBase& mMutexRef;
};

class LockGuardForRead
{
public:
	LockGuardForRead(MutexRW&);
	~LockGuardForRead();
private:
	MutexRW& mMutexRef;
};

//////////////////////

class Condition
{
public:
	Condition(Mutex&);
	~Condition();
	
	int wait();
	int timedWait(int sec, int nsec);
	int timedWait(int msec);
	int signal();
	int broadcast();
private:
	pthread_cond_t mCondition;
	Mutex& mMutexRef;
};

//////////////////////


// To use: inherit from Thread and implement entry()
//   These functions that are NOT (ironically) thread-safe:
//    create()
//    run()
//    cancel()
//    detach()
//    join()
class Thread
{
public:
	Thread(int type = PTHREAD_CREATE_JOINABLE);
	~Thread();

// setup	
	int create();
	
// some scheduling contols
	int run();
	int cancel(); // Use at your own risk.
				  // It is far safer to tell a thread to exit through
				  // syncronized flags and check-points.
				  
	int detach();
	int join(void** exitDataStore = NULL);

// identification
	bool isSelf();
	int getID();

	bool operator==(const Thread&);

protected:
	virtual void entry() = 0;  // override this

// functions that only make sense to be called from within	
	int yield();  // careful not to loop on yield.  will nail system.
	int sleep(int msec);
	void exit(void* exitData);

private:
	// Function run by pthread_create
	static void* _runHelper(void*);
	
	// Called by _runHelper.  Halts the thread until
	// signalled by run().
	void _startingGate();
	
	
	///// Data:

	pthread_t mThread;
	
	int mType;  // attached or detached?
	bool mGo;

	Mutex mSleepMutex;  // mutex *must be* before condition for
	                    // inititialization issues in c_tor
	Condition mSleepCondition;
	
	Mutex mRunMutex;  // mutex *must be* before condition for
					  // inititialization issues in c_tor
	Condition mRunCondition;
};

#endif