/*
 *  stateMachine.cpp
 *  Roomba
 *
 *  Created by Glenn Elliott on 9/7/08.
 *  Copyright 2008 __MyCompanyName__. All rights reserved.
 *
 */

#include <iostream>

#include "stateMachine.h"

#include "createoi.h"

using namespace std;



Timer::Timer():
    Thread(),
    mTimerMutex(),
    mTimerCondition(mTimerMutex),
    mCurrentDelay(0)
{
}

void Timer::start(int32_t milliseconds)
{
    mTimerMutex.lock();
    mCurrentDelay = milliseconds;
    mTimerCondition.signal();
    mTimerMutex.unlock();
}

void Timer::stop()
{
    mTimerMutex.lock();
    mCurrentDelay = 0;
    mTimerCondition.signal();
    mTimerMutex.unlock();        
}


void Timer::entry()
{
    for(;;)
    {
        mTimerMutex.lock();
        if(mCurrentDelay > 0)
        {
            cout<<"Starting timeout for: "<<dec<<mCurrentDelay<<endl;
            int ret = mTimerCondition.timedWait(mCurrentDelay);
            if(ret != 0)
            {
                StateMachine::instance()->signalTimeout();
            }
        }
        else
        {
            cout<<"No timers.  Waiting..."<<endl;
            mTimerCondition.wait();
        }
        mTimerMutex.unlock();
    }
}




StateMachine::StateMachine():
    mWallUpper(4000),
    mWallLower(-1),
    mTerminate(false),
    mEventLoopMutex(),
    mEventLoopCondition(mEventLoopMutex)
{
    mTimer.create();
    mTimer.run();
    
    mRaisedTimeout = false;
}


std::auto_ptr<StateMachine> StateMachine::mInstance;

StateMachine* StateMachine::instance()
{
    if(NULL == mInstance.get())
    {
        mInstance.reset(new StateMachine());
    }
    
    return(mInstance.get());
}


void StateMachine::setInitialState(State* firstState)
{    
    mCurrentState.reset(firstState);
}


void StateMachine::go()
{
    _stateChange(mCurrentState.get(), true);
    
    usleep(250000);  // 1/4 second sleep
    
    while(!mTerminate)  // event loop!
    {
        int sensorVal;
        
        
        if(!mTerminate)
        {
            sensorVal = readSensor(SENSOR_CHARGING_STATE);
            //cout<<"Charging state: "<<sensorVal<<endl;
            if((sensorVal > 0) && (sensorVal < 5))
            {
                cout<<"Charging Event!\t\tValue = "<<dec<<sensorVal<<endl;
                _stateChange(mCurrentState->onCharging());
            }
        }
        
        
        if(!mTerminate)
        {
            sensorVal = getBumpsAndWheelDrops();
            if((sensorVal != 0x52) && (sensorVal != 0xFF) && (sensorVal & 0x03))
            {
                cout<<"Bumper Event!\t\tValue = "<<hex<<sensorVal<<endl;
                _stateChange(mCurrentState->onBump(sensorVal));
            }
        }
        
        
        if(!mTerminate)
        {
            sensorVal = readSensor(SENSOR_WALL_SIGNAL);
            //cout<<"wall sensor says: "<<sensorVal<<endl;
            if(sensorVal <= 4095)
            {  
                if(sensorVal > mWallUpper)
                {
                    cout<<"Wall Over Threshold Event!\t\tValue = "<<sensorVal<<endl;
                    _stateChange(mCurrentState->onWallOverThreshold(sensorVal));
                }
                
                if(sensorVal < mWallLower)
                {
                    cout<<"Wall Under Threshold Event!\t\tValue = "<<sensorVal<<endl;
                    _stateChange(mCurrentState->onWallUnderThreshold(sensorVal));
                }
            }
        }
        
        
        if(!mTerminate)
        {
            sensorVal = readSensor(SENSOR_INFRARED);
            {
                if((sensorVal >= 242) && (sensorVal <= 254))
                {
                    cout<<"IR Event!\t\tValue = "<<hex<<sensorVal<<endl;
                    _stateChange(mCurrentState->onIR(sensorVal));
                }
                else
                {
                    //cout<<"IR Event!\t\tValue = "<<hex<<0<<endl;
                    _stateChange(mCurrentState->onIR(sensorVal));                
                }
            }
        }
        
        
        if(!mTerminate)
        {
            if(mRaisedTimeout)
            {
                cout<<"Timeout Event!"<<endl;
                mRaisedTimeout = false;
                _stateChange(mCurrentState->onTimeout());
            }
        }
        
        mEventLoopMutex.lock();
        mEventLoopCondition.timedWait(100);
        mEventLoopMutex.unlock();
    }
}

void StateMachine::stop()
{
    if(NULL != mCurrentState.get())
    {
        mCurrentState->onExit();
    }
    
    _stateChange(new Shutdown());
}

void StateMachine::terminate()
{
    mTerminate = true;
    
    mEventLoopMutex.lock();
    mEventLoopCondition.signal();
    mEventLoopMutex.unlock();
}

void StateMachine::_stateChange(State* newState, bool redo)
{
    //enableDebug();
    
    
    if(redo || (newState != mCurrentState.get()))
    {
        State* nextState(newState);
        do
        {
            if(nextState != mCurrentState.get())
            {
                mCurrentState.reset(nextState);
            }
            
            nextState = mCurrentState->onEnter();
            
        } while(nextState != mCurrentState.get());
    }
    
    //disableDebug();
}


void StateMachine::setTimeout(int32_t milliseconds)
{
    if(milliseconds != 0)
    {
        mTimer.start(0);  // clear any old timers
    }
    mTimer.start(milliseconds);
}

void StateMachine::setWallUpperThreshold(int32_t upper)
{
    mWallUpper = upper;
}

void StateMachine::setWallLowerThreshold(int32_t lower)
{
    mWallLower = lower;
}

void StateMachine::signalTimeout()
{
    mEventLoopMutex.lock();
    mRaisedTimeout = true;
    mEventLoopMutex.unlock();
}

