/*
 *  states.cpp
 *  Roomba
 *
 *  Created by Glenn Elliott on 9/7/08.
 *  Copyright 2008 __MyCompanyName__. All rights reserved.
 *
 */

#include <iostream>
#include <cmath>
#include <deque>

#include "states.h"
#include "stateMachine.h"

#include "createoi.h"

using namespace std;

namespace
{
    int SUCCESS = 0;
}

Initial::Initial(const string& serialPort):
    mSerialPort(serialPort)
{
  
}

State* Initial::onEnter()
{   
    int ret = startOI(const_cast<char*>(mSerialPort.c_str()));
    if(SUCCESS == ret)
    {
        ret = setBaud(BAUD57600);
        if(SUCCESS == ret)
        {
        }
    }
    
    mOk = (SUCCESS == ret);     

    if(mOk)
    {
        return(new RandomWalk());
    }
    else
    {
        return(this);
    }
}





State* Shutdown::onEnter()
{
    cout<<"Shutdown::onEnter():  Shutting down..."<<endl;
    stopOI();
    StateMachine::instance()->terminate();
    return(this);
}




State* RandomWalk::onEnter()
{
    cout<<"RandomWalk::onEnter"<<endl;
    // set up timeout?
    
    StateMachine::instance()->setWallUpperThreshold(WallFollow::LOWER_THESH+1);
    
    // pick a random direction and go.
    _waggle(MAX_SPEED);
    
    return(this);
}



State* RandomWalk::onBump(int bumperState)
{
    cout<<"RandomWalk::onBump"<<endl;
    
    _stopWaggle();
    
    // back up a little
    getDistance(); // clear the distance.
    driveDistance(-MAX_SPEED, 0, -20, 0); // back up 2 cm
    
    int16_t newDegree(0);

    if((bumperState & 0x01) && (bumperState & 0x02)) // both
    {
        cout<<"both!"<<endl;
        
        do
        {
            newDegree = (int16_t)90 + (int16_t)(rand()%90);
        }while(newDegree == 0);
    }
    else if(bumperState & 0x01)  // right
    {
        cout<<"right!"<<endl;
        
        do
        {        
            newDegree = (int16_t)(rand()%90);
        }while(newDegree == 0);
    }
    else  // left
    {
        cout<<"left!"<<endl;
        
        do
        {        
            newDegree = -(int16_t)(rand()%90);
        }while(newDegree == 0);
    }

    cout<<"Attempting to turn: "<<dec<<newDegree<<endl;
    
    // turn
    getAngle();  // clear the angle.   
    turn(100, (newDegree > 0) ? 1 : -1, newDegree, 0);
    
    // go ahead
    _waggle(MAX_SPEED);

    return(this);
}


void RandomWalk::_waggle(int16_t speed)
{
    //getDistance(); // clears distance measure
    
    StateMachine::instance()->setTimeout(2000);
    mWaggling = true;

    mWaggleDirection = true;
    drive(MAX_SPEED, WAGGLE_ARC);
}


State* RandomWalk::onTimeout()
{
    if(mWaggling)
    {
        cout<<"waggling...";
        
        getAngle();  // clear the angle.
        if(mWaggleDirection)
        {
            cout<<" +"<<endl;
            drive(MAX_SPEED, WAGGLE_ARC);
        }
        else
        {
            cout<<" -"<<endl;
            drive(MAX_SPEED, -WAGGLE_ARC);
        }
        mWaggleDirection = !mWaggleDirection;
    }
    
    return(this);
}

void RandomWalk::_stopWaggle()
{
    mWaggling = false;
    StateMachine::instance()->setTimeout(0);
}


State* RandomWalk::onWallOverThreshold(int wallSignalStrength)
{
    cout<<"Found a wall!!  Changing to WallFollow."<<endl;
    _stopWaggle();
    return(new WallFollow(wallSignalStrength));
}


State* RandomWalk::onIR(int irState)
{
    if(Dock::isBaseStation(irState))
    {
        cout<<"RandomWalk::onIR(): Found an IR signal!  Changing to Dock."<<endl;
        _stopWaggle();
        
        return(new Dock(irState));
    }
    return(this);
}

State* RandomWalk::onCharging()
{
    return(new Shutdown());
}








State* WallFollow::onEnter()
{
    mLost = false;
    
    StateMachine::instance()->setWallUpperThreshold(UPPER_THESH);
    StateMachine::instance()->setWallLowerThreshold(LOWER_THESH);
    
    drive(0,0);  // stop the car.
    
    //disableDebug();
    mLastWallSignal = _maximizeWallSensor(mInitialWallStrength);
    //enableDebug();
    
    drive(MAX_SPEED, WALLFOLLOW_ARC);
    
    return(this);
}



int16_t WallFollow::_maximizeWallSensor(int16_t currentStrength, int16_t increment)
{  
    cout<<"WallFollow::_maximizeWallSensor(): attempting to improve "<<currentStrength<<"."<<endl;
    
    // check termination cases
    if(currentStrength > 100)
    {
        cout<<"WallFollow::_maximizeWallSensor(): Good enough!"<<endl;
        return(currentStrength);
    }
    else if(currentStrength <= 2)
    {
        cout<<"WallFollow::_maximizeWallSensor(): Too poor to try! = "<<currentStrength<<endl;
        return(currentStrength);
    }
    
    if(increment < 5)
    {
        cout<<"WallFollow::_maximizeWallSensor(): Too small of increment to continue!"<<endl;
        return(currentStrength);
    }
    
    
    // search out the best angle.

    getAngle();  // clear the angle.   
    turn(100, 1, increment, 0);    
    
    int wallSensorPlus = readSensor(SENSOR_WALL_SIGNAL);
    
    if(wallSensorPlus > currentStrength)
    {
        currentStrength = _maximizeWallSensor(wallSensorPlus, increment/2);
    }
    else
    {
        getAngle();  // clear the angle.   
        turn(100, -1, -2*increment, 0);
    
        int wallSensorMinus = readSensor(SENSOR_WALL_SIGNAL);
    
        if(wallSensorMinus > currentStrength)
        {
            currentStrength = _maximizeWallSensor(wallSensorMinus, increment/2);
        }
        else
        {
            // go back to original position.
            getAngle();  // clear the angle.   
            turn(100, 1, increment, 0);
            currentStrength = readSensor(SENSOR_WALL_SIGNAL);  // refresh strength
            //currentStrength = _maximizeWallSensor(currentStrength, increment/2);
        }
    }
    
    
    return(currentStrength);
}

State* WallFollow::onWallOverThreshold(int wallSignalStrength)
{
    cout<<"WallFollow::onWallOverThreshold(): Overthreshold."<<endl;
    
    mLastWallSignal = wallSignalStrength;
    
    if(mLost)
    {
        cout<<"WallFollow::onWallOverThreshold(): I am found!"<<endl;
        
        mLost = false;
        
        StateMachine::instance()->setTimeout(0);
        
        mLastWallSignal = _maximizeWallSensor(wallSignalStrength);
        
        drive(MAX_SPEED, WALLFOLLOW_ARC);
    }
    else
    {
        cout<<"WallFollow::onWallOverThreshold(): Ignoring because I'm already good."<<endl;
        StateMachine::instance()->setTimeout(0); // just in case
    }
    
    return(this);
}


State* WallFollow::onWallUnderThreshold(int wallSignalStrength)
{
    cout<<"WallFollow::onWallUnderThreshold(): Underthreshold."<<endl;
    
    if(!mLost)
    {
        if(mLastWallSignal - wallSignalStrength > 20) // corner?
        {
            cout<<"WallFollow::onWallUnderThreshold(): BIG DROP!!!.  Prev = "<<mLastWallSignal<<"  New = "<<wallSignalStrength<<endl;
            
            //mLastWallSignal = wallSignalStrength;
            
            drive(MAX_SPEED, -200);
            
            StateMachine::instance()->setTimeout(LOSTWALL_TIMEOUT);      
        }
        else  // normal counter measures.
        {
            //disableDebug();
            mLastWallSignal = _maximizeWallSensor(wallSignalStrength);
            //enableDebug();
            
            if(mLastWallSignal < LOWER_THESH)
            {
                cout<<"WallFollow::onWallUnderThreshold(): I'm lost!!!!"<<endl;
                
                getAngle();  // clear the angle.   
                turn(100, 1, 15, 0);  // attempt to angle back towards the wall
                
                StateMachine::instance()->setTimeout(LOSTWALL_TIMEOUT);
                mLost = true;
            }
            
            drive(MAX_SPEED, WALLFOLLOW_ARC);
        }
        
        
    }
    else
    {
        cout<<"WallFollow::onWallUnderThreshold(): Ignoring because I'm lost."<<endl;
        mLastWallSignal = wallSignalStrength;
    }
    
    return(this);
}


State* WallFollow::onTimeout()
{
    cout<<"WallFollow::onTimeout(): Lost track of the wall.  Doing random walk."<<endl;
    
    // we've lost the wall.  search for it again.
    return(new RandomWalk());
}


State* WallFollow::onBump(int bumperState)
{
    cout<<"WallFollow::onBump"<<endl;
    
    int16_t newDegree(0);
    
    // back up a little
    getDistance(); // clear the distance.
    driveDistance(-MAX_SPEED, 0, -20, 0); // back up 2 cm
    
    
    
    
    static int bumpCount[2] = {0};  // two second history
    static int lastUpdate = time(0);
    
    int currentTime = time(0);
    if(currentTime - lastUpdate <= 2)
    {
        if(++bumpCount[0] + bumpCount[1]  > 4)
        {
            cout<<"WallFollow::onBump: BUMP COUNT TOO HIGH!!"<<endl;
            
            newDegree = 180;
            bumperState = 0; // clear the bumber state so not to interefere with below.  i'm too lazy to restructure the code.
        }
    }
    else if(currentTime - lastUpdate <= 1)
    {
        swap(bumpCount[0], bumpCount[1]);
        bumpCount[0] = 0;
    }
    else
    {
        bumpCount[0] = 0;
        bumpCount[1] = 0;
    }
    lastUpdate = time(0);
    
    
    
    
    if((bumperState & 0x01) && (bumperState & 0x02)) // both
    {
        newDegree = 90;
    }
    else if(bumperState & 0x01)  // right
    {
        newDegree = 20;
    }
    else if(bumperState & 0x02)  // left
    {
        newDegree = -20;
    }
    
    cout<<"Attempting to turn: "<<dec<<newDegree<<endl;
    
    // turn
    getAngle();  // clear the angle.   
    turn(100, (newDegree > 0) ? 1 : -1, newDegree, 0);
    
    // go ahead
    drive(MAX_SPEED, WALLFOLLOW_ARC);  // drive straight
    
    return(this);
}


State* WallFollow::onIR(int irState)
{
    if(Dock::isBaseStation(irState))
    {
        cout<<"WallFollow::onIR(): Found an IR signal!  Changing to Dock."<<endl;
        StateMachine::instance()->setTimeout(0);
        
        return(new Dock(irState));
    }
    return(this);
}


State* WallFollow::onCharging()
{
    return(new Shutdown());
}



















State* Dock::onEnter()
{
    cout<<"Dock::onEnter()"<<endl;
    
    mLastIR = mInitialIR;
    
    mDocking = false;
    
    //disableDebug();
    //mLastIR = _maximizeIRSensor(mInitialIR);
    //enableDebug();
    
    //StateMachine::instance()->setTimeout(DOCKING_TIME_LIMIT);
    
    drive(DOCKING_SPEED, 0);
    
    return(this);
}

State* Dock::onTimeout()
{
    cout<<"Dock::onTimeout()"<<endl;
    
    if(mLastIR == 0)
    {
        StateMachine::instance()->setTimeout(0);
        return(new RandomWalk());
    }
    return(this);
}

State* Dock::onBump(int bumperState)
{       
    cout<<"Dock::onBump"<<endl;
    
    int16_t backUpDistance = -100;
    
    int16_t backupArc = 0;
    if((bumperState & 0x01) && !(bumperState & 0x02))
    {
        backupArc = -500;
    }
    else if(!(bumperState & 0x01) && (bumperState & 0x02))
    {
        backupArc = 500;
    }
        
    
    if(mLastIR == RGFF)
    {
        cout<<"Dock::onBump():  POSSIBLE DOCKING?"<<endl;
        
        getDistance(); // clear the distance.
        driveDistance(300, 0, 30, 0);  // skoot up the docking station.
        
        // i might be on the docking station.
        drive(0,0);     // stop &
        enterPassiveMode();
        //usleep(500000); // stay still for half a second.
        sleep(2);
        
        int chargeState = readSensor(SENSOR_CHARGING_STATE);
        cout<<"Dock::onBump():  Charge state is = "<<dec<<chargeState<<endl;
        if((chargeState > 0) && (chargeState < 5))
        {
            return(onCharging());
        }
        else
        {
            enterSafeMode();
            backUpDistance = -300;
        }
    }
    
    
    
    int16_t newDegree(0);
    
    // back up a little
    getDistance(); // clear the distance.
    driveDistance(-MAX_SPEED, backupArc, backUpDistance, 0); // back up 2 cm
    
    
    
    
    static int bumpCount[2] = {0};  // two second history
    static int lastUpdate = time(0);
    
    int currentTime = time(0);
    if(currentTime - lastUpdate <= 2)
    {
        if(++bumpCount[0] + bumpCount[1]  > 6)
        {
            cout<<"Dock::onBump: BUMP COUNT TOO HIGH!!"<<endl;
            
            newDegree = 180;
            bumperState = 0; // clear the bumber state so not to interefere with below.  i'm too lazy to restructure the code.
        }
    }
    else if(currentTime - lastUpdate <= 1)
    {
        swap(bumpCount[0], bumpCount[1]);
        bumpCount[0] = 0;
    }
    else
    {
        bumpCount[0] = 0;
        bumpCount[1] = 0;
    }
    lastUpdate = time(0);
    
    
    
    
    if((bumperState & 0x01) && (bumperState & 0x02)) // both
    {
        newDegree = 45;
    }
    else if(bumperState & 0x01)  // right
    {
        newDegree = 10;
    }
    else if(bumperState & 0x02)  // left
    {
        newDegree = -10;
    }
    
    cout<<"Attempting to turn: "<<dec<<newDegree<<endl;
    
    // turn
    getAngle();  // clear the angle.   
    turn(100, (newDegree > 0) ? 1 : -1, newDegree, 0);
    
    // go ahead
    drive(DOCKING_SPEED, 0);  // drive straight
    
    return(this);    
}


State* Dock::onIR(int irState)
{   
    float irStateRank = _rank(irState);
    float lastRank = _rank(mLastIR);
    
    cout<<"Dock::onIR(): old rank = "<<lastRank<<"\tnew rank = "<<irStateRank<<endl;
    
    if(irStateRank > lastRank)
    {
        mLastIR = irState;
        
        // todo: push back the timer???
    }
    else if(irStateRank < lastRank)  // things got worse.  see if we can get back on track.
    {
        int16_t increment;
        if(lastRank <= 1)
        {
            increment = 90;
        }
        else if(lastRank <= 2)
        {
            increment = 90;
        }
        else
        {
            increment = 90;
        }
        
        int16_t tempIR = _maximizeIRSensor(irState, increment);  // try to find something better
        
        if (tempIR == mLastIR)
        {
            getDistance(); // clear the distance.
            driveDistance(MAX_SPEED, 0, 20, 0);  // skoot forward to make up for scanning iterations.
        }
        
        if(_rank(tempIR) < lastRank)  // still stinks
        {
            mLastIR = tempIR;
        }
        else
        {
            mLastIR = tempIR;
        }
    }
    else if(irState != mLastIR)      // no change in rank, but there was a change.
    {
        int16_t newDegree;
        
        if((irState == GREEN) && (mLastIR == RED))
        {
            newDegree = 90;
        }
        else if((irState == RED) && (mLastIR == GREEN))
        {
            newDegree = -90;
        }
        else if((irState == GREEN_FF) && (mLastIR == RED_FF))
        {
            newDegree = 90;
        }
        else //if((irState == RED_FF) && (mLastIR == GREEN_FF))
        {
            newDegree = -90;
        }
        
        getAngle();  // clear the angle.   
        turn(100, (newDegree > 0) ? 1 : -1, newDegree, 0);          
    }
    
    if(mLastIR == 0)
    {
        cout<<"Dock::onIR(): AHHH!! WE'VE LOST THE IR!!!"<<endl;
        StateMachine::instance()->setTimeout(10);
    }
    else
    {
        StateMachine::instance()->setTimeout(0);
    }
    
    //if(RGFF == mLastIR)
    //{
    //    drive(DOCKING_SPEED, 0);  // drive straight at half speed
    //}
    //else
    //{      
        drive(DOCKING_SPEED, 0);  // drive straight
    //}
    
    return(this);
}


State* Dock::onCharging()
{
    cout<<"Dock::onCharging"<<endl;
    return(new Shutdown());
}


int16_t Dock::_maximizeIRSensor(int16_t currentIR, int16_t increment)
{
    cout<<"Dock::_maximizeIRSensor(): attempting to improve "<<hex<<currentIR<<"."<<endl;
    
    // base cases
    if(increment < 10)
    {
        cout<<"Dock::_maximizeIRSensor(): Too small of increment to continue!"<<endl;
        return(currentIR);
    }
    
    if(_rank(currentIR) == _rank(RGFF))
    {
        cout<<"Dock::_maximizeIRSensor(): Good enough!"<<endl;
        return(currentIR);
    }
    else if(_rank(currentIR) == 0.0)
    {
        cout<<"Dock::_maximizeIRSensor(): Too poor to try!"<<endl;
        return(currentIR);
    }
    
    
    // search out the best angle.
    
    getAngle();  // clear the angle.   
    turn(100, 1, increment, 0);    
    
    int16_t irSensorPlus = readSensor(SENSOR_INFRARED);
    
    //if(irSensorPlus > currentIR)
    if(_rank(irSensorPlus) >  _rank(currentIR))
    {
        currentIR = _maximizeIRSensor(irSensorPlus, increment/2);
    }
    else
    {
        getAngle();  // clear the angle.   
        turn(100, -1, -2*increment, 0);
        
        int16_t irSensorMinus = readSensor(SENSOR_INFRARED);
        
        if(_rank(irSensorMinus) >  _rank(currentIR))
        {
            currentIR = _maximizeIRSensor(irSensorMinus, increment/2);
        }
        else
        {
            // go back to original position.
            getAngle();  // clear the angle.   
            turn(100, 1, increment, 0);
            //currentIR = readSensor(SENSOR_INFRARED);  // refresh strength
            
            currentIR = _maximizeIRSensor(currentIR, increment/2);
        }
    }
    
    return(currentIR);
}

bool Dock::isBaseStation(int16_t ir)
{
    switch(ir)
    {
        case FORCE_FIELD:
        case GREEN:
        case GREEN_FF:
        case RED:
        case RED_FF:
        case RED_GREEN:
        case RGFF:
            return(true);
        default:
            return(false);
    }
}

float Dock::_rank(int16_t ir)
{
    switch(ir)
    {
        case FORCE_FIELD:
            return 1.5;
        case GREEN:
            return 1.0;
        case GREEN_FF:
            return 2.5;
        case RED:
            return 1.0;
        case RED_FF:
            return 1.5;
        case RED_GREEN:
            return 2.0;
        case RGFF:
            return 3.0;
        default:
            return 0.0;
    }
}














